/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

/*
 * ModbusServer V2:
 * ========================================================================================================================================================================================================================================================
 * This is an alternative version of the ModbusServer class that uses an alternative method to deal with duplicate inputs.
 *
 * The regular version of ModbusServer protects against duplicate actions by comparing current holding register input against the previous holding register input; if the input has changed, do as the new value dictates.
 * As a result it's possible for valid duplicate inputs to go ignored; for example multiple error clears in a row, or Power/FRQ values if the actual on-screen value were changed by some other means of communication.
 *
 * The Alternate version of ModbusServer protects against duplicate actions by comparing current holding register against the related value in the current input register
 * This ensures that meaningful inputs get through regardless of what the last thing was that you wrote to the holding register.
 *
 * The Alternate version of ModbusServer's success hinges on the presence of the TX bit, and the fact that it is supposed to change every single write cycle
 * The TX bit ensures that everytime a bulk of values is written, there is always a change in the content and the dataWritten signal of QModbusTcpServer is raised properly for all values written to the ModbusServer.
 * If this weren't the case, valid duplicate inputs on the holding register could potentially be completely ignored. (For example after a Reset, or if someone interacted with inputs on the touch display or through another means of communication).
 * ========================================================================================================================================================================================================================================================
 */

#include "modbusserver_v2.h"
#include "miscellaneous.h"
#include <QDebug>
#include <QFile>
#include <QNetworkInterface>
#include <QRandomGenerator>
#include <QUrl>

ModbusServer::ModbusServer(QString modbus_file):mbServer(nullptr)             //mbServer = nullptr
{
	QFile file(modbus_file);
	if(!file.open(QIODevice::ReadOnly))
	{
		qDebug() << "Modbus File unavailable!" << file.errorString() << "\n" << file.fileName();
	}
	else
	{
		/* Read the file and save values to configmap */
		while(!file.atEnd())
		{
			QString line;
			line = file.readLine();
			line.replace("\t","");
			line.replace("\r","");
			line.replace("\n","");

			if (!(line.startsWith("#") || line.isEmpty()))
			{
				 QStringList list = line.split(QRegExp("\\s*=\\s*"));
				 if(list.count() == 2)
				 {
					modbus_map.insert(list.at(0), list.at(1));
				 }
				 else
				 {
					qDebug() << "line syntax incorrect: " << list << "ignored line";
				 }
			}
		}
	}
	file.close();
}

ModbusServer::~ModbusServer()
{
	if (mbServer)
	{
		mbServer->disconnectDevice();
	}
	delete mbServer;
}

void ModbusServer::Initialize()
{
	mbServer = new QModbusTcpServer(this);
	lastServer = new QModbusTcpServer(this);

	//Primary Modbus Map
	QModbusDataUnitMap reg;
//	reg.insert(QModbusDataUnit::Coils, { QModbusDataUnit::Coils, 0, 100 });
//	reg.insert(QModbusDataUnit::DiscreteInputs, { QModbusDataUnit::DiscreteInputs, 0, 100 });
	reg.insert(QModbusDataUnit::InputRegisters, { QModbusDataUnit::InputRegisters, 8192, 10000 });
	reg.insert(QModbusDataUnit::HoldingRegisters, { QModbusDataUnit::HoldingRegisters, 24576, 25000 });
	mbServer->setMap(reg);

	//Secondary modbus map for making comparisons to prevent duplicate inputs
//	QModbusDataUnitMap last_regs;
////	last_regs.insert(QModbusDataUnit::Coils, { QModbusDataUnit::Coils, 0, 100});
////	last_regs.insert(QModbusDataUnit::DiscreteInputs, { QModbusDataUnit::DiscreteInputs, 0, 100 });
////	last_regs.insert(QModbusDataUnit::InputRegisters, { QModbusDataUnit::InputRegisters, 8192, 10000});
//	last_regs.insert(QModbusDataUnit::HoldingRegisters, { QModbusDataUnit::HoldingRegisters, 24576, 25000});
//	lastServer->setMap(last_regs);

	connect(mbServer, &QModbusServer::dataWritten,
			this, &ModbusServer::Holding_Register_Update_Handler);
	connect(mbServer, &QModbusServer::stateChanged,
			this, &ModbusServer::onConnectedStateChanged);
	connect(mbServer, &QModbusServer::errorOccurred,
			this, &ModbusServer::handleDeviceError);
}

#warning why do I even exist?
void ModbusServer::handleDeviceError(QModbusDevice::Error newError)
{
	if (newError == QModbusDevice::NoError || !mbServer)
	{
		return;
	}

	qDebug() << mbServer->errorString();
}

void ModbusServer::onConnectedStateChanged(int state)
{
	QString address =  mbServer->connectionParameter(QModbusDevice::NetworkAddressParameter).toString();
	int port = mbServer->connectionParameter(QModbusDevice::NetworkPortParameter).toInt();
	QString error = mbServer->errorString();
	if (state == QModbusDevice::UnconnectedState)
	{
		emit signal_modbus_connected_enable(false, address, port, error);
	}
	else if (state == QModbusDevice::ConnectedState)
	{
		emit signal_modbus_connected_enable(true, address, port, error);
	}
}

void ModbusServer::set_listen_enable(bool enable)
{
	if (enable)
	{
		QString mbAddress;

		//if address is defined in modbus.txt use that one, otherwise use a valid IPv4 address or localhost address as a last resort.
		if (!modbus_map.contains("modbus_slave_address"))
		{
			QList<QHostAddress> ipAddressesList = QNetworkInterface::allAddresses();
			for (int i = 0; i < ipAddressesList.size(); ++i)
			{
				/* use the first non-localhost IPv4 address */
				if (ipAddressesList.at(i) != QHostAddress::LocalHost && ipAddressesList.at(i).toIPv4Address())
				{
					mbAddress = ipAddressesList.at(i).toString();
					break;
				}
			}
			if (mbAddress.isEmpty())	// if we did not find one, use IPv4 localhost
			{
				mbAddress = QHostAddress(QHostAddress::LocalHost).toString();
			}
		}
		else
		{
			mbAddress = modbus_map.value("modbus_slave_address");
		}

		const QUrl url = QUrl::fromUserInput(mbAddress + ":" + modbus_map.value("modbus_slave_port"));
		mbServer->setConnectionParameter(QModbusDevice::NetworkAddressParameter, url.host());
		mbServer->setConnectionParameter(QModbusDevice::NetworkPortParameter, url.port());
		mbServer->setServerAddress(modbus_map.value("modbus_slave_ID").toInt());

		if (!mbServer->connectDevice())
		{
			qDebug() << "Connect failed: " + mbServer->errorString();
		}
	}
	else
	{
		mbServer->disconnectDevice();
	}
}

/**********************************************************************************************************************************************************************************
 * REGISTERS SET/GETS
 * *******************************************************************************************************************************************************************************/
//Set Register
void ModbusServer::setRegister(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 value)
{
	if (!mbServer)
		return;

	bool ok = true;
	ok = mbServer->setData(reg_type, address, value);

	if (!ok)
	{
		qDebug() << "Could not set register: " + mbServer->errorString();
	}
}

void ModbusServer::setInputRegister(quint16 address, quint16 value)
{
	setRegister(QModbusDataUnit::InputRegisters, address, value);
}

void ModbusServer::setHoldingRegister(quint16 address, quint16 value)
{
	setRegister(QModbusDataUnit::HoldingRegisters, address, value);
}

//Get Register
quint16 ModbusServer::getRegister(QModbusDataUnit::RegisterType reg_type, quint16 address)
{
	if (!mbServer)
		return -1;

	quint16 reg_value = 0;
	if (!mbServer->data(reg_type, address, &reg_value))
	{
		qDebug() << "getRegister: failed to get data |" + mbServer->errorString();
	}

	return (reg_value);
}

quint16 ModbusServer::getInputRegister(quint16 address)
{
	return getRegister(QModbusDataUnit::InputRegisters, address);
}

quint16 ModbusServer::getHoldingRegister(quint16 address)
{
	return getRegister(QModbusDataUnit::HoldingRegisters, address);
}


//Set Register Bit
void ModbusServer::setBit(quint16 &value, quint16 bit_pos, bool enable)
{
	if (enable == true)
	{
		value |= (1 << bit_pos);		//Overwrite to 1;
	}
	else
	{
		value &= ~(1 << bit_pos);	//Overwrite to 0;
	}
}

void ModbusServer::setRegisterBit(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 bit_pos, bool enable)
{
	quint16 reg_value= 0;
	if (mbServer->data(reg_type, address, &reg_value))
	{
		setBit(reg_value, bit_pos, enable);
		setRegister(reg_type, address, reg_value);
	}
}

void ModbusServer::setInputRegisterBit(quint16 address, quint16 bit_pos, bool enable)
{
	setRegisterBit(QModbusDataUnit::InputRegisters, address, bit_pos, enable);
}

void ModbusServer::setHoldingRegisterBit(quint16 address, quint16 bit_pos, bool enable)
{
	setRegisterBit(QModbusDataUnit::HoldingRegisters, address, bit_pos, enable);
}

//Get Register Bit
bool ModbusServer::getRegisterBit(QModbusDataUnit::RegisterType reg_type, quint16 address, quint16 bit_pos)
{
	quint16 reg_value= 0;
	if (!mbServer->data(reg_type, address, &reg_value))
	{
		qDebug() << "getRegisterBit: failed to get data  |" + mbServer->errorString();
	}

	return (reg_value & (1<<bit_pos));
}

bool ModbusServer::getInputRegisterBit(quint16 address, quint16 bit_pos)
{
	return getRegisterBit(QModbusDataUnit::InputRegisters, address, bit_pos);
}

bool ModbusServer::getHoldingRegisterBit(quint16 address, quint16 bit_pos)
{
	return getRegisterBit(QModbusDataUnit::HoldingRegisters, address, bit_pos);
}



/**********************************************************************************************************************************************************************************
 * INPUT REGISTERS
 * *******************************************************************************************************************************************************************************/
void ModbusServer::handler_generator_ready_get(bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 0, enable);
}

void ModbusServer::handler_RF_enable_get(int intrasys_num, bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 1, enable);
}

void ModbusServer::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 4, enable);
}

void ModbusServer::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	#error missing PWM duty cycle and PWM frequency registers
	if (duty_cycle >= 1 && duty_cycle <= 99)	//PWM enabled <> External Triggering Mode disabled.
	{
		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 5, true);
	}
	else
	{
		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 5, false);
	}
}

void ModbusServer::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	power_watt = round(power_watt);		//Handle the rounding properly
	setInputRegister(modbus_map.value("IR_FWD_power_setpoint").toUShort(), (quint16)power_watt);
}

void ModbusServer::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11_loss, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_ratio)
{
	//S11 should be negative; Cap off S11 values to 0 Maximum;
	//Make the negative S11 value absolute (force positive), because registers only support positive values.
	double s11 = PA_s11_loss;
	if (s11 > 0)
		s11 = 0;
	s11 = round((abs(s11) * 1000));

	setInputRegister(modbus_map.value("IR_FWD_power").toUShort(), (quint16)round(PA_power_fwd_watt));
	setInputRegister(modbus_map.value("IR_RFL_power").toUShort(), (quint16)round(PA_power_rfl_watt));
	setInputRegister(modbus_map.value("IR_return_loss").toUShort(), (quint16)s11);		// dB / 1000 -> weird calculation...
}

void ModbusServer::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	// TODO: I'd kind of prefer if we split large values in high + low registers, so that this becomes re-useable in the future.
	quint16 frequency = round(frequency_mhz * 1000);
	setInputRegister(modbus_map.value("IR_actual_frequency").toUShort(), frequency);
}

void ModbusServer::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		QStringList SWP_data = SWP_raw_data.split("\r\n");

		//Remove the OK entry the empty one that comes after.
		QVector<double>	SWP_freq_data;
		QVector<double>	SWP_frw_data;
		QVector<double>	SWP_rfl_data;
		QVector<double>	SWP_s11_dbm_data;
//		QVector<double>	SWP_s11_watt_data;

		SWP_freq_data.resize(SWP_data.count()-2);
		SWP_frw_data.resize(SWP_data.count()-2);
		SWP_rfl_data.resize(SWP_data.count()-2);
		SWP_s11_dbm_data.resize(SWP_data.count()-2);
//		SWP_s11_watt_data.resize(SWP_data.count()-2);

		QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(\\d+.?\\d+),(\\d+.?\\d+)");

		for (int i = 0; i < SWP_data.count() - 2; i++)
		{
			 if (regexp.indexIn(SWP_data.at(i))>=0)
			 {
				  QString string1 = regexp.cap(1);
				  QString string2 = regexp.cap(2);
				  QString string3 = regexp.cap(3);

				  SWP_freq_data[i] = string1.toDouble();
				  SWP_frw_data[i] = string2.toDouble();
				  SWP_rfl_data[i] = string3.toDouble();

				  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
//				  SWP_s11_watt_data[i] = convert_dbm_to_watt(SWP_rfl_data[i]) / convert_dbm_to_watt(SWP_frw_data[i]);

				  /* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
				   * Since we get our power values in dBm and convert to watt, there's no such thing as 0 dbm anyway... therefor in practice divide by zero doesn't occur. */
			 }
		}

		double frequency_best_match = 0;
		double s11_best_match = __INT_MAX__;


		for (int i = 0; i < SWP_s11_dbm_data.size(); i++)
		{
			if (s11_best_match > SWP_s11_dbm_data.at(i))
			{
				s11_best_match = SWP_s11_dbm_data.at(i);		//finds the best S11 match (lowest S11 in dB)
				frequency_best_match = SWP_freq_data.at(i);		//set frequency to frequency of best S11 match
			}
		}

		frequency_best_match = round(frequency_best_match * 1000);

		//S11 should be negative; Cap off S11 values to 0 Maximum;
		//Make the negative S11 value absolute (force positive), because registers only support positive values.
		if (s11_best_match > 0)
			s11_best_match = 0;
		s11_best_match = round(abs(s11_best_match) * 1000);

		double s11_start = SWP_s11_dbm_data.at(0);
		if (s11_start > 0)
			s11_start = 0;
		s11_start = round((abs(s11_start) * 1000));

		double s11_stop = SWP_s11_dbm_data.at(SWP_s11_dbm_data.size()-1);
		if (s11_stop > 0)
			s11_stop = 0;
		s11_stop = round((abs(s11_stop) * 1000));

		//Write Sweep data.
		setInputRegister(modbus_map.value("IR_S11_frequency_best_match").toUShort(), frequency_best_match);
		setInputRegister(modbus_map.value("IR_S11_loss_best_match").toUShort(), s11_best_match);
		setInputRegister(modbus_map.value("IR_S11_loss_start_freq").toUShort(), s11_start);
		setInputRegister(modbus_map.value("IR_S11_loss_stop_freq").toUShort(), s11_stop);

//		//Sweep Complete True
//		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 3, true);
	}
//	else
//	{
//		//Sweep Complete false
//		setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 3, false);
//	}

	//Sweep Running False;
//	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 2, false);
}

void ModbusServer::handler_SWP_settings_get(int instrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	double frq_start = frequency_start * 1000;
	double frq_stop = frequency_stop * 1000;
	double frq_step = frequency_step * 1000;

	setInputRegister(modbus_map.value("IR_SWP_start_frequency").toUShort(), frq_start);
	setInputRegister(modbus_map.value("IR_SWP_stop_frequency").toUShort(), frq_stop);
	setInputRegister(modbus_map.value("IR_SWP_step_frequency").toUShort(), frq_step);
	setInputRegister(modbus_map.value("IR_SWP_power").toUShort(), power_watt);
}

void ModbusServer::handler_temperature_get(int intrasys_num, double temperature)
{
	temperature = round(temperature * 10);
	setInputRegister(modbus_map.value("IR_temperature").toUShort(), (quint16)temperature);
}

void ModbusServer::handler_error_get(int intrasys_num, int error, QStringList error_messages)
{
	quint16 error_high = 0;
	quint16 error_low = 0;
	for (int i = 0; i <= 15; i++)
	{
		setBit(error_low, i, error & (1<<i));
	}
	for (int i = 16; i <= 31; i++)
	{
		setBit(error_high, i-16, error & (1<<i));
	}

	setInputRegister(modbus_map.value("IR_SG_error_high").toUShort(), error_high);
	setInputRegister(modbus_map.value("IR_SG_error_low").toUShort(), error_low);
}

void ModbusServer::handler_datalogging_enable_get(bool enable)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 7, enable);
}

void ModbusServer::handler_PSU_dissipation_get(int intrasys_num, double val)
{
	double dissipation = val;
	if (val < 0)
	{
		dissipation = 0;
	}
	dissipation = round(dissipation);
	setInputRegister(modbus_map.value("IR_PSU_power_dissipation").toUShort(), dissipation);
}

void ModbusServer::handler_SGx_communication_working_get(bool state)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(),11, state);
}

void ModbusServer::handler_PSU_communication_working_get(bool state)
{
	setInputRegisterBit(modbus_map.value("IR_status").toUShort(),12, state);
}

/**********************************************************************************************************************************************************************************
 * HOLDING REGISTERS
 * *******************************************************************************************************************************************************************************/
void ModbusServer::Holding_Register_Update_Handler(QModbusDataUnit::RegisterType table, int address, int size)
{
	for (int i = 0; i < size; ++i)
	{
		quint16 regaddres = address + i;
		quint16 value = 0;
		quint16 last_value = 0;
		QString text = "";

		#error you are here; is it really necessary to create a bunch of extra Input Register entries just so that we can compare holding and input registers? That puts unnecessary burden on Daniele and I could probably make due with some local values for all my stuff instead...
		#error functionally I totally seem to be on the right track here though...
		switch (table)
		{
		case (QModbusDataUnit::HoldingRegisters):
			mbServer->data(QModbusDataUnit::HoldingRegisters, regaddres, &value);
			lastServer->data(QModbusDataUnit::HoldingRegisters, regaddres, &last_value);

			//Bit-based commands register gets per-bit treatment.
			if (regaddres == modbus_map.value("HR_commands").toInt())
			{
				handler_set_commands(value);
			}
			if (regaddres == modbus_map.value("HR_FWD_power_setpoint").toInt())
			{
				handler_set_power_watt(value);
			}
			if (regaddres == modbus_map.value("HR_Frequency_setpoint").toInt())
			{
				handler_set_frequency(value);
			}
//			if (regaddres == modbus_map.value("HR_sweep_start_freq").toInt())
//			{
//				handler_set_SWP_frequency_start(value);
//			}
//			if (regaddres == modbus_map.value("HR_sweep_stop_freq").toInt())
//			{
//				handler_set_SWP_frequency_stop(value);
//			}
//			if (regaddres == modbus_map.value("HR_sweep_step_freq").toInt())
//			{
//				handler_set_SWP_frequency_step(value);
//			}
//			if (regaddres == modbus_map.value("HR_sweep_power").toInt())
//			{
//				handler_set_SWP_power(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_frequency_lower").toInt())
//			{
//				handler_set_DLL_frequency_limit_lower(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_frequency_upper").toInt())
//			{
//				handler_set_DLL_frequency_limit_upper(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_frequency_start").toInt())
//			{
//				handler_set_DLL_frequency_start(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_frequency_step").toInt())
//			{
//				handler_set_DLL_frequency_step(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_threshold").toInt())
//			{
//				handler_set_DLL_threshold(value);
//			}
//			if (regaddres == modbus_map.value("HR_DLL_delay").toInt())
//			{
//				handler_set_DLL_delay(value);
//			}
////			if (regaddres == modbus_map.value("HR_PWM_frequency").toInt())
////			{
////				handler_set_PWM_frequency(value);
////			}
//			if (regaddres == modbus_map.value("HR_PWM_duty_cycle").toInt())
//			{
//				handler_set_PWM_duty_cycle(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_Kp").toInt())
//			{
//				handler_set_PID_Kp(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_Ki").toInt())
//			{
//				handler_set_PID_Ki(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_Kd").toInt())
//			{
//				handler_set_PID_Kd(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_setpoint").toInt())
//			{
//				handler_set_PID_setpoint(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_scaling_factor").toInt())
//			{
//				handler_set_PID_scaling_factor(value);
//			}
//			if (regaddres == modbus_map.value("HR_PID_offset").toInt())
//			{
//				handler_set_PID_offset(value);
//			}

			lastServer->setData(QModbusDataUnit::HoldingRegisters, regaddres, value);
			break;
		default:
			break;
		}
	}
}

/* This function handles the setting of individual bits from the holding register. */
void ModbusServer::handler_set_commands(quint16 val)
{
	for (int i = 0; i < 16; i++)
	{
		switch(i)
		{
		case 0:			//RF enable
			if (getHoldingRegisterBit(modbus_map.value("HR_commands").toInt(), i)
				!= getInputRegisterBit(modbus_map.value("IR_status").toInt(), 1))
			{
				emit signal_set_RF_enable(1, val & (1<<i));
			}
			break;
		case 1:			//Reset
			if (val & (1<<i))
			{
				// Only reset if the value on the relevant bit is 'true';
				qDebug() << "modbus -> Reset";
				emit signal_execute_reset_SGx(1);
			}
			break;
		case 2:			//Clear Error
			if (val & (1<<i))
			{
				// Only Clear Error if the value on the relevant bit is 'true';
				qDebug() << "modbus -> Clear Error";
				emit signal_execute_error_clear(1);
			}
			break;
		case 3:			//Execute Sweep
		{
			if (val & (1<<i))
			{
				quint16 value = 0;
				mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_start_freq").toInt(), &value);
				double start_freq = (double)value / 1000;
				mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_stop_freq").toInt(), &value);
				double stop_freq = (double)value / 1000;
				mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_step_freq").toInt(), &value);
				double step_freq = (double)value / 1000;
				mbServer->data(QModbusDataUnit::HoldingRegisters, modbus_map.value("HR_sweep_power").toInt(), &value);
				double pow_watt = (double)value;

				emit signal_execute_sweep(1, start_freq, stop_freq, step_freq, convert_watt_to_dbm(pow_watt));
			}
		}
			break;
		case 4:			//DLL_enable
			if (getHoldingRegisterBit(modbus_map.value("HR_commands").toInt(), i)
				!= getInputRegisterBit(modbus_map.value("IR_status").toInt(), 4))
			{
				emit signal_set_DLL_enable(1, val & (1<<i));
			}
			break;
		case 5:			//PWM enable
			if (getHoldingRegisterBit(modbus_map.value("HR_commands").toInt(), i)
				!= getInputRegisterBit(modbus_map.value("IR_status").toInt(), 5))
			{
				emit signal_set_PWM_enable(1, val & (1<<i));
			}
			break;
		case 6:			//PID enable
//			if (getHoldingRegisterBit(modbus_map.value("HR_commands").toInt(), i)
//					!= getInputRegisterBit(modbus_map.value("IR_status").toInt(), 6))
//			{
//				emit signal_set_PID_enable(1, newVal & (1<<i));
//			}
			break;
		case 7:
			if (getHoldingRegisterBit(modbus_map.value("HR_commands").toInt(), i)
				!= getInputRegisterBit(modbus_map.value("IR_status").toInt(), 7))
			{
				emit signal_set_datalogging_enable(val & (1<<i));
			}
			break;
			#error missing case 8 Protection board reset
		case 8:
			if (val & (1<<i))
			{
				#error test this real quick yea?
				qDebug() << "modbus -> Reset Protection Board";
				emit signal_execute_reset_protection(1);
			}
			break;
		case 15:		//TX changed
			setInputRegisterBit(modbus_map.value("IR_status").toUShort(), 15, (val & (1<<i)));
			break;
			default:
				break;
		}
	}
}

void ModbusServer::handler_set_power_watt(quint16 value)
{
	if(getHoldingRegister(modbus_map.value("HR_FWD_power_setpoint").toInt()) != getInputRegister(modbus_map.value("IR_FWD_power_setpoint").toInt()))
	{
		emit signal_set_power_watt(1, value);
	}
}

void ModbusServer::handler_set_frequency(quint16 value)
{
	if(getHoldingRegister(modbus_map.value("HR_Frequency_setpoint").toInt()) != getInputRegister(modbus_map.value("IR_actual_frequency").toInt()))
	{
		double frequency = (double)value / 1000;
		emit signal_set_frequency(1, frequency);
	}
}

void ModbusServer::handler_set_SWP_frequency_start(quint16 value)
{

	double frequency = (double)value / 1000;
	emit signal_set_SWP_frequency_start(1, frequency);
}

void ModbusServer::handler_set_SWP_frequency_stop(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_SWP_frequency_stop(1, frequency);
}

void ModbusServer::handler_set_SWP_frequency_step(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_SWP_frequency_step(1, frequency);
}

void ModbusServer::handler_set_SWP_power(quint16 value)
{
	emit signal_set_SWP_power(1, value);
}

void ModbusServer::handler_set_DLL_frequency_limit_lower(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_DLL_frequency_limit_lower(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_limit_upper(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_DLL_frequency_limit_upper(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_start(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_DLL_frequency_start(1, frequency);
}

void ModbusServer::handler_set_DLL_frequency_step(quint16 value)
{
	double frequency = (double)value / 1000;
	emit signal_set_DLL_frequency_step(1, frequency);
}

void ModbusServer::handler_set_DLL_threshold(quint16 value)
{
	double threshold = (double)value / 1000;
	emit signal_set_DLL_threshold(1, threshold);
}

void ModbusServer::handler_set_DLL_delay(quint16 value)
{
	emit signal_set_DLL_delay(1, value);
}

void ModbusServer::handler_set_PWM_frequency(quint16 value)
{
	emit signal_set_PWM_frequency(1,(double)value);
}

void ModbusServer::handler_set_PWM_duty_cycle(quint16 value)
{
	emit signal_set_PWM_duty_cycle(1,(int)value);
}

void ModbusServer::handler_set_PID_Kp(quint16 value)
{
	double kVal = (double) value / 100;
	emit signal_set_PID_Kp(1,(double)kVal);
}

void ModbusServer::handler_set_PID_Ki(quint16 value)
{
	double kVal = (double) value / 100;
	emit signal_set_PID_Ki(1,(double)kVal);
}

void ModbusServer::handler_set_PID_Kd(quint16 value)
{
	double kVal = (double) value / 100;
	emit signal_set_PID_Kd(1,(double)kVal);
}

void ModbusServer::handler_set_PID_setpoint(quint16 value)
{
	emit signal_set_PID_setpoint(1,(double)value);
}

void ModbusServer::handler_set_PID_scaling_factor(quint16 value)
{
	double scale = (double) value / 100;
	emit signal_set_PID_scaling(1,(double)scale);
}

void ModbusServer::handler_set_PID_offset(quint16 value)
{
	emit signal_set_PID_offset(1,(double)value);
}
